<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class CodeSendBuyer implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $order;

    /**
     * Create a new job instance.
     */
    public function __construct($order)
    {
        $this->order = $order;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $order = $this->order;
        $orderStatus = 1; // Default to complete
        $hasStockShort = false;

        \Log::info("Processing order: " . $order->utr);

        foreach ($order->orderDetails as $detail) {
            $service = $detail->detailable;
            if (!$service) {
                \Log::warning("No service found for order detail: " . $detail->id);
                continue;
            }

            // Get available codes
            $codeLists = $service->codes()->where('status', 1)->take($detail->qty)->get(['id', 'status', 'passcode']);
            $sendCodeList = $codeLists->pluck('passcode');
            $stock_short = max(0, $detail->qty - count($sendCodeList));

            \Log::info("Order detail: {$detail->id}, Requested: {$detail->qty}, Found: " . count($sendCodeList));

            // Update order detail
            $detail->card_codes = $sendCodeList;
            $detail->stock_short = $stock_short;
            $detail->status = ($stock_short == 0) ? 1 : 3;
            $detail->save();

            if ($stock_short > 0) {
                $hasStockShort = true;
                $orderStatus = 3; // Set to pending if any stock is short
                \Log::warning("Stock short for order: {$order->utr}, detail: {$detail->id}, short: {$stock_short}");
            }

            // Mark codes as used
            foreach ($codeLists as $code) {
                \Log::info("Marking code {$code->id} as used");
                $code->status = 0; // Mark as inactive/used
                $code->save();     // Save the change
                $code->delete();   // Then delete if needed
            }

            $this->addedSellList($service->card()->select(['id', 'sell_count'])->first());
        }

        // Update order status
        $order->status = $orderStatus;
        $order->save();

        \Log::info("Order {$order->utr} processed. Final status: {$orderStatus}");

        // Force complete orders that should be complete
        if (!$hasStockShort && $order->status != 1) {
            \Log::info("Forcing order {$order->utr} to complete status");
            $order->status = 1;
            $order->save();
        }
    }

    public function addedSellList($card): void
    {
        if ($card) {
            $card->sell_count += 1;
            $card->save();
        }
    }
}





